//	TorusGamesRootController.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesRootController.h"
#import "TorusGamesMatteView.h"
#import "TorusGamesGraphicsViewiOS.h"
#import "GeometryGamesModel.h"
#import "GeometryGamesUtilities-iOS.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesLocalization.h"
#import "GeometryGamesSound.h"
#import "GeometryGamesBevelImage-iOS.h"
#import <QuartzCore/QuartzCore.h>	//	for Core Animation
#import <sys/utsname.h>				//	for uname()


//	If the matte underlaps the toolbar,
//	the toolbar will absorb some of the matte's greyish blue color.
//	Otherwise it would be more logical not to have it underlap.
#define MATTE_UNDERLAPS_TOOLBAR

//	If desired, these two constants could be unified into a single
//
//		#define BEVEL_WIDTH_PHONE	9
//
//	For now, though, I'm happy to leave the bevel
//	a little thinner on iPhone and a little thicker on iPad.
#define BEVEL_WIDTH_PHONE				 6
#define BEVEL_WIDTH_PAD					12

//	Parameters for spinning the board while the game resets.
#define RESET_ANIMATION_DURATION		1.0		//	seconds
#define RESET_ANIMATION_DURATION_JIGSAW	1.5		//	seconds
#define RF1								0.25	//	reset shrinkage factor at 90° and 270°
#define RF2								0.0625	//	reset shrinkage factor at 180°

#ifdef SCIENCE_CENTER_VERSION
//	After how many seconds of inactivity should
//	Torus Games reset itself and re-display Choose a Game?
//#warning restore default timeout period
#define INACTIVITY_TIMEOUT_PERIOD		30.0	//	in seconds
#endif


//	Privately-declared properties and methods
@interface TorusGamesRootController()

- (void)loadMatteView;
- (void)loadMessageView;
- (void)loadGameViews;
- (void)loadKeyboardView;
- (void)loadToolbarView;

- (void)setMessageFontFaceForCurrentLanguage;
- (void)setMessageFontSizeForGame:(GameType)aGame;
- (void)refreshToolbarContent;
- (void)userTappedToolbarButton:(id)sender;

- (void)resetGameWithAnimationForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue;
- (void)resetGameWithSpinningSquareAnimationForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue;
- (void)resetGame:(NSTimer *)aTimer;
- (void)resetGameForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue;
- (void)halfwayPoint:(NSTimer *)aTimer;
- (void)ensureMazeVisibility;

- (bool)changeLanguage:(const Char16 [3])aTwoLetterLanguageCode;

- (void)refreshMessageCentering;

#ifdef SCIENCE_CENTER_VERSION
- (void)inactivityTimerFired:(id)sender;
- (void)resetExhibit;
#endif

@end


@implementation TorusGamesRootController
{
	TorusGamesMatteView		*itsMatteView;
	UILabel					*itsMessageView;
	UIView					*itsGameEnclosureView;
	UIImageView				*itsBevelView;
	TorusGamesKeyboardView	*itsKeyboardView;
	UIToolbar				*itsToolbar;

	UIBarButtonItem			*itsResetGameButton,
							*itsChangeGameButton,
							*itsFlexibleSpace,
							*itsOptionsButton,
							*itsLanguageButton,
							*itsHelpButton;

#ifdef SCIENCE_CENTER_VERSION
	//	itsInactivityTimer keeps a strong reference to its target,
	//	namely this TorusGamesRootController.
	//	If this TorusGamesRootController also
	//	kept a strong reference to itsInactivityTimer,
	//	we'd have a strong reference cycle and neither object
	//	would ever get released.  So keep a weak reference instead.
	//	The run loop will keep a strong reference to the timer
	//	as long as its valid, so there's no danger
	//	of it getting released too soon.
	NSTimer	* __weak	itsInactivityTimer;	//	retained by run loop
	
	bool				itsUserIsActivelyPlaying,
						itsKeepPlayingMessageIsUp;
#endif
}


- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil
{
	self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
	if (self != nil)
	{
#ifdef SCIENCE_CENTER_VERSION
		itsKeepPlayingMessageIsUp	= false;
		itsUserIsActivelyPlaying	= false;
#endif
	}
	return self;
}

- (void)dealloc
{
	//	The TorusGamesRootController never gets released,
	//	therefore this -dealloc never gets called.

	[self changeGame:GameNone];

#ifdef SCIENCE_CENTER_VERSION
	//	itsInactivityTime retains this TorusGamesRootController,
	//	so we're unlikely to arrive here until itsActivityTimer == nil.
	//	(Indeed, we're unlikely to ever arrive here at all!)
	[itsInactivityTimer invalidate];	//	OK even if itsInactivityTimer == nil
#endif
}

- (BOOL)prefersStatusBarHidden
{
	return NO;
}

- (void)viewDidLoad
{
	[super viewDidLoad];

	[self loadMatteView];	//	itsMatteView will contain itsMessageView, itsGameEnclosureView and itsKeyboardView.
	[self loadMessageView];
	[self loadGameViews];
	[self loadKeyboardView];
	
	[self loadToolbarView];	//	Load itsToolbar last, on top of itsMatteView.
}

- (void)loadMatteView
{
	ModelData	*md	= NULL;
	GameType	theGame;
	CGRect		theMatteFrame;

	//	[[self view] bounds] includes both the region at the top
	//	that will get covered by the transparent status bar
	//	and the region at the bottom that will get covered
	//	by the toolbar.
	
	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];
	
//#warning Relying on viewSafeAreaInsetsDidChange is risky (see details in code).
//	Relying on viewSafeAreaInsetsDidChange works, but it's
//	a brittle solution that could break in the future.
//	(Even now, if we suppress the status bar on a notchless iPhone,
//	viewSafeAreaInsetsDidChange doesn't get called.)
//	Fortunately the SwiftUI version of this app
//	will avoid this problem altogether, which is why
//	I'm not going to fix it in this legacy code.

	theMatteFrame = [[self view] bounds];	//	viewSafeAreaInsetsDidChange may override
	itsMatteView = [[TorusGamesMatteView alloc]
		initWithDelegate:	self
		frame:				theMatteFrame	//	frame may get overridden in viewSafeAreaInsetsDidChange
		game:				theGame];
	[[self view] addSubview:itsMatteView];
	[itsMatteView setAutoresizingMask:(UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight)];
	[itsMatteView setBackgroundColor:[UIColor
		//	colorWithDisplayP3Red:green:blue:alpha: takes gamma-encoded color components
		colorWithDisplayP3Red:	gMatteColorGammaP3[0]
		green:					gMatteColorGammaP3[1]
		blue:					gMatteColorGammaP3[2]
		alpha:					1.0]];
}

- (void)loadMessageView
{
	//	TorusGamesMatteView's -layoutSubviews will override itsMessageView's frame.
	itsMessageView = [[UILabel alloc] initWithFrame:(CGRect){{0.0, 0.0}, {1.0, 1.0}}];
	[itsMatteView addMessageView:itsMessageView];
	[itsMessageView setTextAlignment:NSTextAlignmentCenter];
	[itsMessageView setAdjustsFontSizeToFitWidth:YES];
	[itsMessageView setMinimumScaleFactor:0.25];
	[itsMessageView setNumberOfLines:0];
	[itsMessageView setOpaque:NO];
	[itsMessageView setBackgroundColor:[UIColor colorWithRed:0.0 green:0.0 blue:0.0 alpha:0.0]];
//[itsMessageView setBackgroundColor:[UIColor colorWithRed:1.00 green:0.00 blue:0.00 alpha:1.0]];
}

- (void)loadGameViews
{
	CGRect			theGameEnclosureFrame;
	CGRect			theBevelFrame;
	unsigned int	theBevelWidth;
	CGRect			theGameFrame;

	//	TorusGamesMatteView's -layoutSubviews will override itsGameEnclosureView's frame.
	theGameEnclosureFrame = (CGRect){{0.0, 0.0}, {64.0, 64.0}};
	itsGameEnclosureView = [[UIView alloc] initWithFrame:theGameEnclosureFrame];
	[itsMatteView addGameEnclosureView:itsGameEnclosureView];

	if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPad)
		theBevelWidth = BEVEL_WIDTH_PAD;
	else
		theBevelWidth = BEVEL_WIDTH_PHONE;
	
	theBevelFrame	= (CGRect){CGPointZero, theGameEnclosureFrame.size};
	itsBevelView	= [[UIImageView alloc] initWithFrame:theBevelFrame];
	[itsGameEnclosureView addSubview:itsBevelView];
	[itsBevelView setImage:
		StretchableBevelImage(	gMatteColorGammaP3,
								theBevelWidth,
								[[UIScreen mainScreen] scale],
								UIDisplayGamutP3)];
	[itsBevelView setAutoresizingMask:(UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight)];

	theGameFrame = (CGRect)
		{
			{
				theBevelWidth,
				theBevelWidth
			},
			{
				theGameEnclosureFrame.size.width  - 2*theBevelWidth,
				theGameEnclosureFrame.size.height - 2*theBevelWidth
			}
		};

	itsMainView = [[TorusGamesGraphicsViewiOS alloc]
						initWithModel:	itsModel
						frame:			theGameFrame];
	[itsMainView setUpGraphics];

	[itsGameEnclosureView addSubview:itsMainView];
	[itsMainView setExclusiveTouch:YES];	//	disables all controls while touches are in progress
	[itsMainView setAutoresizingMask:(UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight)];
}

- (void)loadKeyboardView
{
	GameType	theGame;
	ModelData	*md	= NULL;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	//	TorusGamesMatteView's -layoutSubviews will override itsKeyboardView's frame.
	itsKeyboardView = [[TorusGamesKeyboardView alloc] initWithDelegate:self frame:(CGRect){{0.0, 0.0}, {1.0, 1.0}}];
	[itsMatteView addKeyboardView:itsKeyboardView];
	[itsKeyboardView refreshWithLanguage:
		(theGame == Game2DCrossword ? GetCurrentLanguage() : u"--")];
//[itsKeyboardView setBackgroundColor:[UIColor colorWithRed:0.0 green:1.0 blue:0.0 alpha:1.0]];
}

- (void)loadToolbarView
{
//#warning Relying on viewSafeAreaInsetsDidChange is risky (see details in code).
//	Relying on viewSafeAreaInsetsDidChange works, but it's
//	a brittle solution that could break in the future.
//	(Even now, if we suppress the status bar on a notchless iPhone,
//	viewSafeAreaInsetsDidChange doesn't get called.)
//	Fortunately the SwiftUI version of this app
//	will avoid this problem altogether, which is why
//	I'm not going to fix it in this legacy code.

	itsToolbar = [[UIToolbar alloc] initWithFrame:CGRectMake(0.0, 0.0, 32.0, 0.0)];
	[itsToolbar sizeToFit];	//	Sets itsToolbar's height to the default height for the device.
	[[self view] addSubview:itsToolbar];
	[itsToolbar setAutoresizingMask:(UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleTopMargin)];
	
	itsResetGameButton  = [[UIBarButtonItem alloc]
		initWithTitle:	LocalizationNotNeeded(@"dummy")	//	refreshToolbarContent will set title or image, as appropriate
		style:			UIBarButtonItemStylePlain
		target:			self
		action:			@selector(userTappedToolbarButton:)];

	itsChangeGameButton = [[UIBarButtonItem alloc]
		initWithTitle:	LocalizationNotNeeded(@"dummy")	//	refreshToolbarContent will set title or image, as appropriate
		style:			UIBarButtonItemStylePlain
		target:			self
		action:			@selector(userTappedToolbarButton:)];
	
	itsFlexibleSpace	= [[UIBarButtonItem alloc]
		initWithBarButtonSystemItem:	UIBarButtonSystemItemFlexibleSpace
		target:							self
		action:							@selector(userTappedToolbarButton:)];

	itsOptionsButton	= [[UIBarButtonItem alloc]
		initWithImage:	[UIImage imageNamed:@"Toolbar Buttons/Options"]
		style:			UIBarButtonItemStylePlain
		target:			self
		action:			@selector(userTappedToolbarButton:)];

	itsLanguageButton	= [[UIBarButtonItem alloc]
		initWithImage:	[UIImage imageNamed:@"Toolbar Buttons/Language"]
		style:			UIBarButtonItemStylePlain
		target:			self
		action:			@selector(userTappedToolbarButton:)];

	itsHelpButton		= [[UIBarButtonItem alloc]
		initWithImage:	[UIImage imageNamed:@"Toolbar Buttons/Help"]
		style:			UIBarButtonItemStylePlain
		target:			self
		action:			@selector(userTappedToolbarButton:)];

	[itsToolbar	setItems:	@[
								itsResetGameButton,
								itsChangeGameButton,
								itsFlexibleSpace,
								itsOptionsButton,
								itsLanguageButton,
								itsHelpButton
							]
				animated:	NO];
	
	[self refreshToolbarContent];
}

- (void)viewWillLayoutSubviews
{
	CGFloat	theTopMargin,
			theBottomMargin;
	
	//	Comments in StarterCodeRootController's viewWillLayoutSubviews
	//	summarize the order in which views get laid out.

	//	Even though a translucent toolbar overlays the content view,
	//	-safeAreaInsets doesn't know about it,
	//	because the toolbar is our own custom toolbar,
	//	not a UIKit toolbar provided by a UINavigationController.
	//
	theTopMargin	= [[self view] safeAreaInsets].top;
#ifdef MATTE_UNDERLAPS_TOOLBAR
	theBottomMargin	= [[self view] safeAreaInsets].bottom + [itsToolbar frame].size.height;
#else
	theBottomMargin	= 0.0;
#endif

	[itsMatteView setMarginTop:theTopMargin bottom:theBottomMargin];
}


#ifdef SCIENCE_CENTER_VERSION

#pragma mark -
#pragma mark starting/stopping inactivity timer

- (void)viewDidAppear:(BOOL)animated
{
	[super viewDidAppear:animated];
	
	//	At launch on iPhone (but not on iPad) we get one, in this order
	//
	//		viewWillDisappear	(immediately)
	//		viewDidAppear		(immediately after the viewDidDisappear call)
	//		viewDidAppear		(much laster, after the user dismisses the Choose a Game presented view)
	//
	//	If we didn't explicitly clear itsInactivityTimer here, the first NSTimer
	//	would be left running even after we created and installed the second one.
	//
	if (itsInactivityTimer != nil)			//	Shouldn't occur, but does!
		[itsInactivityTimer invalidate];	//	Deallocates the timer and thus clears our weak reference to it.

	itsInactivityTimer = [NSTimer
		scheduledTimerWithTimeInterval:	INACTIVITY_TIMEOUT_PERIOD
		target:							self	//	retained;  released when timer gets invalidated
		selector:						@selector(inactivityTimerFired:)
		userInfo:						nil
		repeats:						YES];
}

- (void)viewWillDisappear:(BOOL)animated
{
	//	Invalidate itsInactivityTimer, which will in turn release it
	//	and automatically clear our weak reference to it.
	[itsInactivityTimer invalidate];
	
	[super viewWillDisappear:animated];
}

#endif	//	SCIENCE_CENTER_VERSION


#ifdef MAKE_SCREENSHOTS

#pragma mark -
#pragma mark automate screenshots

static bool	gViewHasBeenRedrawn	= false;


- (void)viewDidAppear:(BOOL)animated
{
	NSString	*thePreferredLanguage;	//	user's preferred language at the iOS level

	static BOOL	theScreenshotsHaveBeenStarted	= false;
	
	[super viewDidAppear:animated];

	thePreferredLanguage = GetPreferredLanguage();

	//	The first time the view appears (and only the first time!)
	//	start a script running to generate all the various screenshots.
	if ( ! theScreenshotsHaveBeenStarted )
	{
		theScreenshotsHaveBeenStarted = YES;

		dispatch_after(
			dispatch_time(DISPATCH_TIME_NOW, (int64_t)(0 * NSEC_PER_SEC)),
			dispatch_queue_create(
				"org.geometrygames.4DMazeScreenshotQueue",
				DISPATCH_QUEUE_SERIAL),
			^{
				NSArray<NSString *>	*theLanguages;
				unsigned int		theLanguageIndex,
									theScreenshotIndex;

				//	The screenshots for Japanese, Simplified Chinese and Traditional Chinese
				//	must get rendered while that language is set not only within Torus Games,
				//	but also in the Xcode scheme (which in effect means that Torus Games
				//	sees that langauge as the de facto iOS system language), so that the system
				//	will provide the correct font for the Unihan characters.
				//
				//	The screenshots for Arabic (and in principle for Farsi and Hebrew)
				//	can get rendered while the Xcode scheme's language is set to Arabic,
				//	to provide the desired right-to-left layout.
				//
				//	The screenshots for all other languages may get rendered
				//	while the Xcode scheme's language is set to a Western language
				//	(typically I use English).
				//
				if ([thePreferredLanguage isEqualToString:@"ja"])
					theLanguages = @[@"ja"];
				else
				if ([thePreferredLanguage isEqualToString:@"zs"])
					theLanguages = @[@"zs"];
				else
				if ([thePreferredLanguage isEqualToString:@"zt"])
					theLanguages = @[@"zt"];
				else
				if ([thePreferredLanguage isEqualToString:@"ar"])
					theLanguages = @[];	//	= @[@"ar", @"fa", @"he"];	if any of those translations were present
				else	//	translations that require no special treatment
					theLanguages = @[@"de", @"el", @"en", @"es", @"fi", @"fr", @"it", @"ko", @"nl", @"pt", @"ru", @"vi"];
					//	Known bug with this approach:  Any UIBarButtonItems created
					//	with UIBarButtonSystemItemDone or UIBarButtonSystemItemCancel
					//	will appear in whatever language the app is running in (typically "en")
					//	rather than the language that gets set in the loop below.
					//	The easiest workaround, if we're willing to clutter up the source code,
					//	is to create the UIBarButtonItem using initWithTitle:GetLocalizedTextAsNSString(u"Done")
					//	instead of initWithBarButtonSystemItem:UIBarButtonSystemItemDone.
					//	A more general workaround would be to run the app in each language separately,
					//	just like we're already doing for ar, ja, zh-Hans and zh-Hant,
					//	but running each Western language separately as well.
					//	On the one hand, it would be a bit tedious to manually run them
					//	one language at a time, but on the other hand, when I write
					//	the SwiftUI versions of the Geometry Games apps, this could
					//	keep the new apps' internal structure simple by not having
					//	to bypass SwiftUI's built-in localization mechanism.

				for (theLanguageIndex = 0; theLanguageIndex < [theLanguages count]; theLanguageIndex++)
				{
					NSString	*theTwoLetterLanguageCode;

					theTwoLetterLanguageCode = theLanguages[theLanguageIndex];
					
					//	Use dispatch_sync, not dispatch_async.
					dispatch_sync(dispatch_get_main_queue(),
					^{
						Char16	theTwoLetterLanguageCodeAsCString[3] =
								{
									[theTwoLetterLanguageCode characterAtIndex:0],
									[theTwoLetterLanguageCode characterAtIndex:1],
									0
								};
						
						[self changeLanguage:theTwoLetterLanguageCodeAsCString];
					});
					
					GameType	theGameListWestern[10] =
								{
									Game2DPool, Game2DJigsaw, Game2DChess, Game2DCrossword, Game2DApples,
									Game2DMaze, Game2DGomoku, Game2DWordSearch, Game2DTicTacToe, Game3DMaze
								},
								theGameListJaKoVi[10] =
								{
									//	In Japan, Korea and Vietnam, people are more familiar
									//	with Gomoku and than with Western Chess.
									Game2DGomoku, Game2DJigsaw, Game2DMaze, Game2DPool, Game2DCrossword,
									Game2DApples, Game2DChess, Game2DWordSearch, Game2DTicTacToe, Game3DMaze
								},
								theGameListZh[10] =	//	for both zh-Hans and zh-Hant
								{
									//	In China, people are less familiar with Gomoku
									//	than elsewhere, but aren't so familiar with Western Chess either.s
									Game2DPool, Game2DJigsaw, Game2DMaze, Game2DCrossword, Game2DApples,
									Game2DWordSearch, Game2DChess, Game2DTicTacToe, Game2DGomoku, Game3DMaze
								},
								*theGameList;
					
					if ([theTwoLetterLanguageCode isEqualToString:@"ja"]
					 || [theTwoLetterLanguageCode isEqualToString:@"ko"]
					 || [theTwoLetterLanguageCode isEqualToString:@"vi"])
					{
						theGameList = theGameListJaKoVi;
					}
					else
					if ([theTwoLetterLanguageCode isEqualToString:@"zs"]	//	zh-Hans
					 || [theTwoLetterLanguageCode isEqualToString:@"zt"])	//	zh-Hant
					{
						theGameList = theGameListZh;
					}
					else
					{
						theGameList = theGameListWestern;
					}
					
					//	theScreenshotIndex steps through theGameList.
					for (theScreenshotIndex = 0; theScreenshotIndex < BUFFER_LENGTH(theGameListWestern); theScreenshotIndex++)
					{
						GameType	theGame;

						//	On iPhone, five of the screenshots show no text,
						//	and in principle they could be the same for all languages.
						//	However... those five are scattered out among the overall set of ten,
						//	so, given that the App Store makes us upload the screenshots
						//	for each language separately anyhow, it seems simpler
						//	to go ahead and generate them separately for each language,
						//	even though some are redundant, to avoid having to mess with
						//	the screenshot order when I upload them to the App Store.
						//	In fact... the screenshot order is different in ja, ko and vi
						//	than it is in the Western languages, and different still
						//	in zs and zt (meaning zh-Hans and zh-Hant), which is all
						//	the more reason to generate a full set of screenshots
						//	for each language separately.
						//
						//	If I ever change my mind about this, I can imitate
						//	the MAKE_SCREENSHOTS code from one of the other Geometry Games apps
						//	to render those five language-independent screenshots
						//	with "theLanguageCode = theLanguagesAll".
						
						theGame = theGameList[theScreenshotIndex];

						@autoreleasepool
						{
							//	Use dispatch_sync, not dispatch_async,
							//	to guarantee that the game will be reset
							//	before we start waiting for a redraw.
							dispatch_sync(dispatch_get_main_queue(),
							^{
								[self resetGameForPurpose:ResetWithNewGame value:theGame];
							});
							
							//	A call to animationTimerFired -- including its subcall
							//	to refreshAllViews -- cannot overlap the block above,
							//	because both run on the main thread.  So to guarantee correctness
							//	it suffices to set gViewHasBeenRedrawn to false
							//	and then wait for refreshAllViews to set it to true.
							//
							//		Technical note:  I would have used a dispatch_semaphore_t
							//		here, but we'd need to devise a way to prevent
							//		animationTimerFired from cranking the value of the semaphore
							//		up to some huge value, because we start waiting on it here.
							//		I'm sure a solution could be devised, but it's not worth
							//		wasting time over for this private-use-only code.
							//
							gViewHasBeenRedrawn = false;
							while ( ! gViewHasBeenRedrawn )
								[NSThread sleepForTimeInterval:0.125];
							
							//	In at least one case, the text in the message box wasn't fully drawn
							//	when the screenshot got taken (perhaps because the message box animates
							//	in and out of view according to whether the current game needs it or not).
							//	So let's allow an extra half-second to make sure the message box
							//	becomes fully visible before we make the screenshot.
							[NSThread sleepForTimeInterval:2.0];	//	0.5 sec is OK on device;  use 2.0 sec on simulator

							//	Save a screenshot.
							//	Use dispatch_sync, not dispatch_async.
							dispatch_sync(dispatch_get_main_queue(),
							^{
								const Char16	*theCurrentLanguage;
//								UIView			*theScreenshotView;
								NSString		*theScreenSize;
								UIImage			*theScreenshotImage;
								const Char16	*theLanguageCode;
								
							//	static Char16	theLanguagesAll[4] = u"all";

								theCurrentLanguage = GetCurrentLanguage();
								NSLog(@"drawing %C%C-%d", theCurrentLanguage[0], theCurrentLanguage[1], theScreenshotIndex);

								switch ((unsigned int)[[UIScreen mainScreen] bounds].size.height)
								{
									//	It's OK to provide screenshots only for
									//	the largest iPhone size with and without a home button, and
									//	the largest iPad size with and without a home button.
								//	case  480:	theScreenSize = @"3-5-inch";		break;
								//	case  568:	theScreenSize = @"4-0-inch";		break;
								//	case  667:	theScreenSize = @"4-7-inch";		break;
									case  736:	theScreenSize = @"phone-HomeBtn";	break;	//	5.5"
									case  896:	theScreenSize = @"phone-HomeInd";	break;	//	6.5"
								//	case 1024:	theScreenSize = @"pad";				break;
									case 1366:	//	12.9", iPad Pro 2nd or 4th generation
										if ([[[UIApplication sharedApplication] keyWindow] safeAreaInsets].bottom > 0.0)
											theScreenSize = @"pad-HomeInd";
										else
											theScreenSize = @"pad-HomeBtn";
										break;
									default:	theScreenSize = @"UNKNOWN-SIZE";	break;
								}
								
	//							theScreenshotView = [[UIScreen mainScreen] snapshotViewAfterScreenUpdates:YES];
	//							UIGraphicsBeginImageContextWithOptions([theScreenshotView bounds].size, YES, 0.0);
	//							[theScreenshotView drawViewHierarchyInRect:[theScreenshotView bounds] afterScreenUpdates:YES];

								UIGraphicsBeginImageContextWithOptions([[[UIApplication sharedApplication] keyWindow] bounds].size, YES, 0.0);
								[[[UIApplication sharedApplication] keyWindow] drawViewHierarchyInRect:[[[UIApplication sharedApplication] keyWindow] bounds] afterScreenUpdates:YES];

								theScreenshotImage = UIGraphicsGetImageFromCurrentImageContext();
								UIGraphicsEndImageContext();

//								if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPhone
//								 && theScreenshotIndex == …)
//								{
//									//	As noted earlier, the first five iPhone screenshots
//									//	contain no text, so we may render each one only once,
//									//	and use it for all the languages.
//									theLanguageCode = theLanguagesAll;
//								}
//								else
								{
									theLanguageCode = GetCurrentLanguage();
								}

								[UIImagePNGRepresentation(theScreenshotImage)
									writeToFile:	GetFullFilePath(
														[NSString stringWithFormat:@"TorusGames-%S-%@-%d",
															theLanguageCode,
															theScreenSize,
															theScreenshotIndex],
														@".png")
									atomically:		YES];
							});

//							//	Dismiss the presented view controller, if any.
//							//	Without animation, of course.
//							//	Use dispatch_sync, not dispatch_async.
//							dispatch_sync(dispatch_get_main_queue(),
//							^{
//								if ([self presentedViewController] != nil)
//									[self dismissViewControllerAnimated:NO completion:nil];
//							});
//
//							//	Caution:  These sleep calls are a complete hack!
//							//	For non-release-quality code, though, they do the job.
//							//	Release-quality code would need to devise some sort
//							//	of scheme to wait/signal a semaphore.
//							[NSThread sleepForTimeInterval:1.0];
						}
					}
				}
				
				dispatch_sync(dispatch_get_main_queue(),
				^{
					[self resetGameForPurpose:	ResetWithNewGame
						  value:				GameNone];
						  
					NSLog(@"done!");
				});
			});
	}
}

- (void)refreshAllViews
{
	[super refreshAllViews];
	
	gViewHasBeenRedrawn = true;
}

#endif	//	MAKE_SCREENSHOTS


#pragma mark -
#pragma mark size change

- (void)viewWillTransitionToSize:(CGSize)size withTransitionCoordinator:(id<UIViewControllerTransitionCoordinator>)coordinator
{
	GeometryGamesModel		*theModel;
	TorusGamesKeyboardView	*theKeyboardView;

	//	Call the GeometryGamesGraphicsViewController's implementation of this method.
	[super viewWillTransitionToSize:size withTransitionCoordinator:coordinator];

	//	Create local references to itsModel and itsKeyboardView
	//	to avoid referring to "self" in the animation block.
	theModel		= itsModel;
	theKeyboardView	= itsKeyboardView;

	//	Submit the animation request.
	[coordinator
		animateAlongsideTransition:^(id<UIViewControllerTransitionCoordinatorContext> context)
		{
			ModelData	*md	= NULL;
			GameType	theGame;
			
			UNUSED_PARAMETER(context);
			
			[theModel lockModelData:&md];
			theGame = md->itsGame;
			[theModel unlockModelData:&md];
			
			//	The Word Search's Chinese poems need to be formatted
			//	differently in landscape and portrait orientations.
			[theModel lockModelData:&md];
			RefreshStatusMessageText(md);
			[theModel unlockModelData:&md];
			
			//	Re-position the keyboard's keys for the new layout.
			if (theGame == Game2DCrossword)
				[theKeyboardView refreshWithLanguage:GetCurrentLanguage()];
		}
	 	completion:^(id<UIViewControllerTransitionCoordinatorContext> context)
		{
		}
	];
}

- (void)viewSafeAreaInsetsDidChange
{
	CGRect			theRootViewBounds;
	UIEdgeInsets	theSafeAreaInsets;
	CGFloat			theToolbarHeight,
					theToolbarOffset;	//	distance from top of toolbar to bottom of view
	CGRect			theMatteFrame,
					theToolbarFrame;

	[super viewSafeAreaInsetsDidChange];

	theRootViewBounds	= [[self view] bounds];
	theSafeAreaInsets	= [[self view] safeAreaInsets];
	theToolbarHeight	= [itsToolbar frame].size.height;
	theToolbarOffset	= theToolbarHeight + theSafeAreaInsets.bottom;	//	allow for home indicator

	//	itsMatteView
	theMatteFrame = theRootViewBounds;
#ifdef MATTE_UNDERLAPS_TOOLBAR
	//	Let itsMatteView fill the whole view,
	//	including the area under the home indicator.
#else
	theMatteFrame.size.height -= theToolbarOffset;
#endif
	[itsMatteView setFrame:theMatteFrame];

	//	itsToolbar
	theToolbarFrame.origin.x	= theRootViewBounds.origin.x;
	theToolbarFrame.origin.y	= theRootViewBounds.size.height - theToolbarOffset;
	theToolbarFrame.size.width	= theRootViewBounds.size.width;
	theToolbarFrame.size.height	= theToolbarHeight;
	[itsToolbar setFrame:theToolbarFrame];
}

- (void)traitCollectionDidChange:(UITraitCollection *)previousTraitCollection
{
	//	Let the superclass take care of any presented views.
	[super traitCollectionDidChange:previousTraitCollection];
	
	//	Replace toolbar text buttons with image buttons, or vice versa, as needed.
	[self refreshToolbarContent];
}


#pragma mark -
#pragma mark message view

- (void)setMessageText:(NSString *)aMessage forGame:(GameType)aGame
{
	UIView	*theMatteView;
	
	//	Adjust itsMessageView's font face for the current language.
	[self setMessageFontFaceForCurrentLanguage];
	
	//	Adjust itsMessageView's font size for the given game.
	[self setMessageFontSizeForGame:aGame];

	//	Write aMessage into itsMessageView.
	[itsMessageView setText:aMessage];
	
	//	When the 3D Maze replaces a nonempty status message with an empty one,
	//	itsMatteView's landscape layout no longer needs to allow space
	//	for itsMessageView.  Otherwise the following call to setNeedsLayout
	//	wouldn't be needed.
	//
	//		Note #1.  Our caller lives in the platform-independent C code
	//		and already holds a lock on the ModelData. If we called
	//		layoutIfNeeded directly here, it too would request a lock
	//		on that same ModelData, resulting in a deadlock.
	//		To avoid that, schedule setNeedsLayout and layoutIfNeeded
	//		to run on the main queue at some point after the current pass
	//		through the run loop is complete.
	//
	//		Note #2.  Copy the instance variable itsMatteView
	//		to the local variable theMatteView, so that the block
	//		will retain only theMatteView, not "self".
	//
	theMatteView = itsMatteView;
	dispatch_async(dispatch_get_main_queue(),
	^{
		[UIView animateWithDuration:0.5 animations:
		^{
			[theMatteView setNeedsLayout];
			[theMatteView layoutIfNeeded];
		}];
	});
}

- (void)setMessageFontFaceForCurrentLanguage
{
	NSString	*theFontName;

	//	The page
	//
	//		http://iosfonts.com
	//
	//	shows which fonts are included in which versions of iOS.
	//
	     if (IsCurrentLanguage(u"ja"))	theFontName = @"HiraKakuProN-W3";
	else if (IsCurrentLanguage(u"ko"))	theFontName = @"AppleGothic";
	else if (IsCurrentLanguage(u"zs"))	theFontName = @"STHeitiSC-Light";
	else if (IsCurrentLanguage(u"zt"))	theFontName = @"STHeitiTC-Light";
	else								theFontName = @"Helvetica";

	if ( ! [theFontName isEqualToString:[[itsMessageView font] fontName]] )
	{
		[itsMessageView setFont:[UIFont
			fontWithName:theFontName
			size:[[itsMessageView font] pointSize]]];
	}
}

- (void)setMessageFontSizeForGame:(GameType)aGame
{
	CGFloat	theOldFontSize,
			theNewFontSize;
	
	theOldFontSize = [[itsMessageView font] pointSize];
	
	switch (aGame)
	{
		case Game2DCrossword:
		case Game2DPool:
			if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPad)
				theNewFontSize = 24.0;
			else	//	iPhone
				theNewFontSize = 20.0;
			break;

		case Game2DWordSearch:
			if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPad)
				theNewFontSize = 24.0;
			else	//	iPhone
				theNewFontSize = 18.0;	//	Chinese poems just barely fit
			break;

		case Game2DApples:
			if ([[UIDevice currentDevice] userInterfaceIdiom] == UIUserInterfaceIdiomPad)
				theNewFontSize = 24.0;
			else	//	iPhone
				theNewFontSize = 16.0;
			break;
		
		case Game3DMaze:
			theNewFontSize = 18.0;
			break;
		
		default:
			theNewFontSize =  8.0;
			break;
	}

	if (theNewFontSize != theOldFontSize)
		[itsMessageView setFont:[[itsMessageView font] fontWithSize:theNewFontSize]];
}

- (void)setMessageColor:(UIColor *)aColor
{
	[itsMessageView setTextColor:aColor];
}


#pragma mark -
#pragma mark toolbar

- (void)refreshToolbarContent
{
	ModelData	*md	= NULL;
	GameType	theGame;
	Char16		*theResetKey,
				*theChangeGameKey;
	
	//	Text labels are clearer than icons for the Reset and Change Game buttons,
	//	but in a horizontally compact layout there typically won't be enough space
	//	for them, especially in space-hungry languages like French and Russian.

	if ([[self traitCollection] horizontalSizeClass] == UIUserInterfaceSizeClassCompact)
	{
		//	Use icons for the Reset and Change Game buttons.

		//	Note:  The Reset icon does *not* need to be mirrored
		//	in right-to-left languages, because even in right-to-left cultures
		//	clocks still turn clockwise.  Moreover, on the page
		//
		//		https://www.google.com/design/spec/usability/bidirectionality.html#bidirectionality-rtl-mirroring-guidelines
		//
		//	the section "When not to mirror" says explicitly that
		//
		//		"A clock icon or a circular refresh or progress indicator
		//		with an arrow pointing clockwise should not be mirrored."
		//
		//
		[itsResetGameButton  setImage:[UIImage imageNamed:@"Toolbar Buttons/ResetGame" ]];
		[itsResetGameButton  setTitle:nil];

		[itsChangeGameButton setImage:[UIImage imageNamed:@"Toolbar Buttons/ChangeGame"]];
		[itsChangeGameButton setTitle:nil];
	}
	else	//	horizontal size class is "regular"
	{
		//	Use text labels for the Reset and Change Game buttons.

		[itsModel lockModelData:&md];
		theGame = md->itsGame;
		[itsModel unlockModelData:&md];
		
		switch (theGame)
		{
			case GameNone:			theResetKey = u"S: Reset";						break;

			case Game2DIntro:		theResetKey = u"S: Reset practice board";		break;
			case Game2DTicTacToe:	theResetKey = u"S: Erase Tic-Tac-Toe board";	break;
			case Game2DGomoku:		theResetKey = u"S: Clear Gomoku board";			break;
			case Game2DMaze:		theResetKey = u"S: New Maze";					break;
			case Game2DCrossword:	theResetKey = u"S: New Crossword puzzle";		break;
			case Game2DWordSearch:	theResetKey = u"S: New Word Search";			break;
			case Game2DJigsaw:		theResetKey = u"S: New Jigsaw puzzle";			break;
			case Game2DChess:		theResetKey = u"S: Reset Chess board";			break;
			case Game2DPool:		theResetKey = u"S: Reset Pool balls";			break;
			case Game2DApples:		theResetKey = u"S: New Apples";					break;

			case Game3DTicTacToe:	theResetKey = u"S: Erase Tic-Tac-Toe board";	break;
			case Game3DMaze:		theResetKey = u"S: New Maze";					break;

			default:				theResetKey = NULL;								break;
		}

		theChangeGameKey = (theGame == GameNone ? u"S: Choose a game" : u"S: Change game");

		[itsResetGameButton  setTitle:GetLocalizedTextAsNSString(theResetKey     )];
		[itsResetGameButton  setImage:nil];

		[itsChangeGameButton setTitle:GetLocalizedTextAsNSString(theChangeGameKey)];
		[itsChangeGameButton setImage:nil];
	}
}

- (void)userTappedToolbarButton:(id)sender
{
	ModelData				*md	= NULL;
	GameType				theGame;
	bool					theHumanVsComputer;
	TopologyType			theTopology;
	unsigned int			theDifficultyLevel;
	ViewType				theViewType;
	UIViewController		*theContentViewController;
	UINavigationController	*theNavigationController;
	NSArray<UIView *>		*thePassthroughViews;

#ifdef SCIENCE_CENTER_VERSION
	[self userIsStillPlaying];
#endif

	[itsModel lockModelData:&md];
	theGame				= md->itsGame;
	theHumanVsComputer	= md->itsHumanVsComputer;
	theTopology			= md->itsTopology;
	theDifficultyLevel	= md->itsDifficultyLevel;
	theViewType			= md->itsViewType;
	[itsModel unlockModelData:&md];
	
	//	Reset Game is the only button that doesn't open a new view.
	if (sender == itsResetGameButton)
	{
		[self resetGameWithAnimationForPurpose:ResetPlain value:0];
		return;
	}

	//	On iOS 8 and later, the UIPopoverPresentationController's setBarButtonItem
	//	automatically adds all of a given button's siblings to the list
	//	of passthrough views, but does not add the given button itself.
	//	So a second tap on the same button will automatically dismiss the popover,
	//	without generating a call to this userTappedToolbarButton method.
	//
	//	Thus when we do get a call to userTappedToolbarButton, we know that
	//	if a popover is already up, it's not this one.  So we may safely dismiss
	//	the already-up popover (if present) and display the newly requested one,
	//	with no risk that it might be the same popover.

	//	If a panel is already present, dismiss it.
	[self dismissViewControllerAnimated:YES completion:nil];
	
	//	Create a table view controller whose content 
	//	corresponds to the button the user touched.
	if (sender == itsChangeGameButton)
	{
		theContentViewController = [[TorusGamesGameChoiceController alloc]
				initWithDelegate:	self
				gameChoice:			theGame
			];
	}
	else
	if (sender == itsOptionsButton)
	{
		theContentViewController = [[TorusGamesOptionsChoiceController alloc]
				initWithDelegate:	self
				humanVsComputer:	theHumanVsComputer
				topology:			theTopology
				difficultyLevel:	theDifficultyLevel
				viewType:			theViewType
				soundEffects:		gPlaySounds
				duringGame:			theGame
			];
	}
	else
	if (sender == itsLanguageButton)
	{
		theContentViewController = [[TorusGamesLanguageChoiceController alloc]
				initWithDelegate:	self
				languageChoice:		GetCurrentLanguage()
			];
	}
	else
	if (sender == itsHelpButton)
	{
		theContentViewController = [[TorusGamesHelpChoiceController alloc]
				initWithDelegate:	self];
	}
	else
	{
		theContentViewController = nil;
	}

	if (theContentViewController != nil)
	{
		//	Present theContentViewController in a navigation controller.
		//
		//	Note:  Even the Preferences Picker gets wrapped in a navigation controller,
		//	to give it a header that doesn't scroll along with the table's contents
		//	(unlike a table header view, which does scroll along with the table's contents).
		//	On iPhone, the nav bar will get a Close button
		//	which we want to keep visible at all times, regardless of scrolling.
		//
		theNavigationController = [[UINavigationController alloc]
			initWithRootViewController:theContentViewController];
		[[theNavigationController navigationBar] setTranslucent:NO];	//	so content won't underlap nav bar

	
		//	In the Torus Games, tapping outside a popover typically dismisses it.
		//	The Help popover is an exception:  the user may experiment
		//	with current game while the Help popover remains visible.
		if (sender == itsHelpButton)
			thePassthroughViews = @[itsMainView];
		else
			thePassthroughViews = nil;

		//	Display theContentViewController, as a popover if possible.
		PresentPopoverFromToolbarButton(self, theNavigationController, sender, thePassthroughViews);
	}
}

- (void)simulateChangeGameButtonTap
{
	[self userTappedToolbarButton:itsChangeGameButton];
}


#pragma mark -
#pragma mark reset animation

- (void)resetGameWithAnimationForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue
	//	aNewValue may be a new GameType, new TopologyType, new difficulty level or new ViewType,
	//	according to aPurpose
{
	ModelData	*md	= NULL;
	GameType	theGame;
	
	//	We've got three ways to reset a game:
	//
	//		-resetGameForPurpose:value:
	//			resets the game immediately, with no animation.
	//
	//		Reset3DGameWithAnimation()
	//			resets a 3D game with a spinning cube (in ViewBasicLarge)
	//			or fade in/out animation (in ViewRepeating),
	//			using Metal directly (no Core Animation).
	//
	//		-resetGameWithSpinningSquareAnimationForPurpose:value:
	//			resets the game with a spinning square animation,
	//			applied to itsGameEnclosureView, using Core Animation.
	//
	//	The present function makes the purely aesthetic decision
	//	of which animation to apply in which circumstances.
	//	This decision may be revised at the programmer's whim,
	//	subject only to the constraint that the second method listed above
	//	applies only to 3D games, not 2D ones.

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];
	
	//	If the old game and/or the new game is GameNone,
	//	don't animate the transition.
	if (theGame == GameNone
	 || (aPurpose == ResetWithNewGame && aNewValue == GameNone))
	{
		[self resetGameForPurpose:aPurpose value:aNewValue];
	}
	else
	//	Some games, for example Game2DChess, will accept
	//	a new difficulty level without interrupting the current game.
	//	In such cases, no animation is needed.
	if (aPurpose == ResetWithNewDifficultyLevel
	 && (	theGame == Game2DGomoku
		 || theGame == Game2DChess
		 || theGame == Game2DPool))
	{
		[self resetGameForPurpose:aPurpose value:aNewValue];
	}
	else
	//	If the game is 3D, and we're not changing to a different game,
	//	then animate the transition with Metal directly.
	//	(Letting Metal animate the change to a different game,
	//	even to another 3D game, would be tricky because halfway
	//	through the animation we'd need to modify the toolbar buttons.)
	if (GameIs3D(theGame) && aPurpose != ResetWithNewGame)
	{
		[itsModel lockModelData:&md];
		Reset3DGameWithAnimation(md, aPurpose, (unsigned int)aNewValue);
		[itsModel unlockModelData:&md];
	}
	else
	//	In all remaining cases, animate the transition using Core Animation.
	{
		[self resetGameWithSpinningSquareAnimationForPurpose:aPurpose value:aNewValue];
	}
}

- (void)resetGameWithSpinningSquareAnimationForPurpose:(ResetPurpose)aPurpose
			value:(unsigned int)aNewValue
				//	aNewValue may be a new GameType, new TopologyType
				//	or new difficulty level, according to aPurpose
{
	ModelData			*md	= NULL;
	GameType			theGame;
	bool				theJigsawWantsExtraSpin;
	CFTimeInterval		theAnimationDuration;
	CAKeyframeAnimation	*theAnimation;
	NSArray				*theResetInfo;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	//	If the user re-selects the game that's already running,
	//	leave that game untouched.  Don't run the animation,
	//	and don't reset the game.
	if (aPurpose == ResetWithNewGame
	 && theGame == (GameType)aNewValue)
	{
		return;
	}

	//	Disable user interaction during the reset.
	[[self view] setUserInteractionEnabled:NO];

	//	Stop the usual animation timer, so that the user won't see
	//	the newly reset game until both
	//
	//		(1)	the reset animation has reached its halfway point
	//	and
	//		(2)	the new game is ready.
	//
	[self stopAnimation];

	//	Start the Reset Game animation running in a separate thread.
	//
	//	Technical Note:  It seems simplest to go straight to Core Animation
	//	and write this as a keyframe animation.  Something similar might
	//	be possible with UIView-based animations, but it's not obvious
	//	that it would be any easier.
	//
	theAnimation = [CAKeyframeAnimation animationWithKeyPath:@"transform"];
	if (@available(iOS 13.0, *))
	{
		//	On newer iDevices we call TorusGamesComputeFunctionMakeJigsawCollages()
		//	which does 16-bit arithmetic and runs plenty fast.
		//	There's no need for the animation to allow an extra rotation.
		theJigsawWantsExtraSpin = false;
	}
	else
	{
		//	On iDevices with A7 or older GPUs, we must call the legacy
		//	function TorusGamesComputeFunctionMakeJigsawCollagesLEGACY(),
		//	which does 32-bit arithmetic.  And such devices are slower
		//	in any case.  So for them the animation must allow an extra rotation
		//	to give the GPU time to finish preparing the collages.
		theJigsawWantsExtraSpin =
		(
			(aPurpose == ResetWithNewGame && aNewValue == Game2DJigsaw)
		 || (aPurpose != ResetWithNewGame &&  theGame  == Game2DJigsaw)
		);
	}
	if (theJigsawWantsExtraSpin)
	{
		//	The Jigsaw Puzzle needs more time to refresh itself than the other apps do,
		//	so give it a 1.5 sec animation, with an extra spin in the middle.
		//	The animation will
		//
		//		begin at time 0,
		//		finish shrinking at time (1/4)*(3/2) = 3/8 sec,
		//		spin until time (3/4)*(3/2) = 9/8 sec,
		//		finish growing at time 3/2 sec.
		//
		//	So it's enough that the puzzle refresh complete within 9/8 sec,
		//	which is about what it takes.  Having the new puzzle appear
		//	just as the board is coming out of the spin (rather than during the spin)
		//	is OK too.
		theAnimationDuration = RESET_ANIMATION_DURATION_JIGSAW;

		[theAnimation setValues:
			@[
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	+1.0,  0.0,  0.0,  0.0,
						 0.0, +1.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 0.0, +RF1,  0.0,  0.0,
						-RF1,  0.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	-RF2,  0.0,  0.0,  0.0,
						 0.0, -RF2,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 0.0, -RF2,  0.0,  0.0,
						+RF2,  0.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 RF2,  0.0,  0.0,  0.0,
						 0.0,  RF2,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 0.0, +RF2,  0.0,  0.0,
						-RF2,  0.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	-RF2,  0.0,  0.0,  0.0,
						 0.0, -RF2,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 0.0, -RF1,  0.0,  0.0,
						+RF1,  0.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	+1.0,  0.0,  0.0,  0.0,
						 0.0, +1.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}]
			]
		];
	}
	else
	{
		//	All the games except the Jigsaw refresh in under half a second,
		//	so they'll be ready by the halfway point of a 1-second animation.
		theAnimationDuration = RESET_ANIMATION_DURATION;

		[theAnimation setValues:
			@[
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	+1.0,  0.0,  0.0,  0.0,
						 0.0, +1.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 0.0, +RF1,  0.0,  0.0,
						-RF1,  0.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	-RF2,  0.0,  0.0,  0.0,
						 0.0, -RF2,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	 0.0, -RF1,  0.0,  0.0,
						+RF1,  0.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}],
				[NSValue valueWithCATransform3D:(CATransform3D)
					{	+1.0,  0.0,  0.0,  0.0,
						 0.0, +1.0,  0.0,  0.0,
						 0.0,  0.0, +1.0,  0.0,
						 0.0,  0.0,  0.0, +1.0	}]
			]
		];
	}
	[theAnimation setDuration:theAnimationDuration];
	[theAnimation setDelegate:nil];
	[[itsGameEnclosureView layer] addAnimation:theAnimation forKey:@"reset game"];
	
	//	Package up the reset information as an NSArray.
	theResetInfo = @[@((NSUInteger)aPurpose), @(aNewValue)];
	
	//	Schedule a timer to fire half way through the reset animation.
	//	Note that if the resetGame call (immediately below) takes longer
	//	than this, we won't get the halfwayPoint message until resetGame
	//	has returned, because those two calls take place in the same thread.
	//
	//		Note:  The run loop will call -halfwayPoint:
	//		on this same thread, which is the main thread.
	//
	[NSTimer	scheduledTimerWithTimeInterval:	(0.5 * theAnimationDuration)
				target:							self
				selector:						@selector(halfwayPoint:)
				userInfo:						nil
				repeats:						NO];
	
	//	The animation won't start until we pass control back to the run loop,
	//	so set a timer to reset the game more-or-less immediately after
	//	we return control to the run loop.
	//
	//		Note #1:  The run loop will call -resetGame:
	//		on this same thread, which is the main thread.
	//
	//		Note #2:  When loading a new Jigsaw puzzle on older versions
	//		of the Torus Games, running on older versions of iOS and on older devices,
	//		we could pass
	//
	//			scheduledTimerWithTimeInterval:0.000001
	//
	//		and the animation would start fine.  With the current set-up,
	//		we must pass an interval of 0.001 seconds or longer
	//		for the animation to start correctly.  Even with 0.0001 seconds,
	//		the animation doesn't start until after the new puzzle has loaded.
	//		For robustness, let's go ahead and pass 0.01 instead of 0.001
	//
	[NSTimer	scheduledTimerWithTimeInterval:0.01
				target:self
				selector:@selector(resetGame:)
				userInfo:theResetInfo
				repeats:NO];
}

- (void)resetGame:(NSTimer *)aTimer
{
	NSArray			*theResetInfo;
	ResetPurpose	thePurpose;
	unsigned int	theNewValue;
	
	theResetInfo	= [aTimer userInfo];
	thePurpose		= (ResetPurpose)[[theResetInfo objectAtIndex:0] unsignedIntValue];
	theNewValue		=               [[theResetInfo objectAtIndex:1] unsignedIntValue];

	[self resetGameForPurpose:thePurpose value:theNewValue];
}

- (void)resetGameForPurpose:(ResetPurpose)aPurpose value:(unsigned int)aNewValue
{
	//	ResetGame() is slow for the jigsaw puzzle,
	//	and re-creating data-dependent textures might be slow
	//	for games like the Crossword and Word Search,
	//	so let's make those calls now, while the animation runs, 
	//	rather than waiting until we want to draw the new game.
	
	//	Call chain
	//
	//	Typically -resetGameWithAnimationForPurpose:value:
	//	sets a timer to call -resetGame:, which then calls
	//	-resetGameForPurpose:value:.
	//
	//	However,  -resetGameWithAnimationForPurpose:value:
	//	may call -resetGameForPurpose:aPurpose value: directly,
	//	in the exceptional case that the old or new game is GameNone.
	//
	//	Either way, -resetGameForPurpose:value: gets called on the main thread,
	//	so even if we modify the ModelData piecemeal here and there,
	//	there's no risk of it getting drawn "mid-stream" (because
	//	on iOS the animation gets done in the main thread).
	
	GameType		theOldGame,
					theNewGame;
	TopologyType	theNewTopology;
	unsigned int	theNewDifficultyLevel;
	ModelData		*md	= NULL;
	
	//	While the animation runs in a separate thread,
	//	and the halfway timer waits, reset the game...
	switch (aPurpose)
	{
		case ResetPlain:

			[itsModel lockModelData:&md];
			ResetGame(md);
			if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
				[((TorusGamesGraphicsViewiOS *)itsMainView)
					refreshRendererTexturesForGameResetWithModelData:md];
			[itsModel unlockModelData:&md];

			break;
		
		case ResetWithNewGame:
		
			//	Let's handle ResetWithNewGame here in the platform-dependent code,
			//	rather than in a platform-independent ChangeGame() function,
			//	so we can update the platform-dependent UI with no need for callbacks.
		
			//	Note the old and new games.
			[itsModel lockModelData:&md];
			theOldGame = md->itsGame;
			theNewGame = (GameType)aNewValue;
			[itsModel unlockModelData:&md];
			
			//	If the user re-selects the game that's already running,
			//	leave that game untouched.  Don't reset it.
			if (theNewGame != theOldGame)
			{
				//	Shut down the old game.
				[itsModel lockModelData:&md];
				ShutDownGame(md);
				[itsModel unlockModelData:&md];
		
				//	Switch to the new game.
				[itsModel lockModelData:&md];
				SetNewGame(md, theNewGame);
				[itsModel unlockModelData:&md];
				
				//	Adapt the layout to the new game's needs.
				//	Do this before calling SetUpGame(), 
				//	so -setMessageText can choose the best font size.
				[itsMatteView refreshLayoutForGame:theNewGame];
		
				//	If the user selected the Crossword, set up the keyboard in the current language.
				//	Otherwise empty the keyboard.
				[itsKeyboardView refreshWithLanguage:
					(theNewGame == Game2DCrossword ? GetCurrentLanguage() : u"--")];
				
				//	Clear the message text.
				[self setMessageText:nil forGame:theNewGame];

				//	Refresh itsMessageView's centering.
				[self refreshMessageCentering];
		
				//	Set up the new game.

				//	Let the game set up game-specific textures.
				[itsModel lockModelData:&md];
				SetUpGame(md);
				if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
					[((TorusGamesGraphicsViewiOS *)itsMainView) refreshRendererTexturesWithModelData:md];
				[itsModel unlockModelData:&md];
				
				//	Update the Reset Game button for the new game,
				//	and update the Change Game button
				//	from "S: Choose a game" to "S: Change game" if necessary.
				[self refreshToolbarContent];

				//	-ensureMazeVisibility is needed only when the Help chooser
				//	has asked for Game3DMaze in a horizontally regular layout
				//	and we want to make sure that the Maze's blue slider and red goal
				//	aren't hidden by the popover itself.
				[self ensureMazeVisibility];
			}
	
			break;
		
		case ResetWithNewTopology:

			theNewTopology = (TopologyType)aNewValue;

			//	Keep the ModelData locked while changing the topology and updating the textures,
			//	to ensure that the animatation thread will always find the ModelData and the textures
			//	in a consistent state.
			[itsModel lockModelData:&md];
			ChangeTopology(md, theNewTopology);	//	always resets the game
			if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
				[((TorusGamesGraphicsViewiOS *)itsMainView) refreshRendererTexturesWithModelData:md];
			[itsModel unlockModelData:&md];

			break;
		
		case ResetWithNewDifficultyLevel:

			theNewDifficultyLevel = (unsigned int)aNewValue;

			[itsModel lockModelData:&md];
			ChangeDifficultyLevel(md, theNewDifficultyLevel);
			if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
				[((TorusGamesGraphicsViewiOS *)itsMainView) refreshRendererTexturesWithModelData:md];
			[itsModel unlockModelData:&md];
			
			break;
	}
	
	//	OK, the game is now reset.
	//
	//	If we're in the middle of an animation (as is typically the case)
	//	then either the halfway timer will have already fired
	//	and the halfwayPoint message will be waiting for us,
	//	or the halfway timer won't yet have fired and we'll wait for it.
}

- (void)halfwayPoint:(NSTimer *)aTimer
{
	UNUSED_PARAMETER(aTimer);

	//	The game has been reset and the animation has reached 
	//	its halfway point (or maybe further), so resume normal updates.
	[self startAnimation];

	//	Re-enable user interaction.
	[[self view] setUserInteractionEnabled:YES];
}

- (void)ensureMazeVisibility
{
	ModelData	*md	= NULL;
	signed int	theMazeSize,
				theDesiredSliderX,
				theOffset[3];

	[itsModel lockModelData:&md];

	//	If the Help chooser has asked for Game3DMaze in a horizontally regular layout,
	//	make sure that the Maze's blue slider and red goal aren't hidden by the popover itself.
	//
	if (md->itsGame == Game3DMaze
	 && [[self traitCollection] horizontalSizeClass] == UIUserInterfaceSizeClassRegular
	 && [[self presentedViewController] isKindOfClass:[UINavigationController class]]
	 && [[[((UINavigationController *)[self presentedViewController]) viewControllers] objectAtIndex:0]
	 					isKindOfClass:[TorusGamesHelpChoiceController class]] )
	{
		GEOMETRY_GAMES_ASSERT(
				md->its3DTilingIntoFrameCell[3][0] == 0.0
			 && md->its3DTilingIntoFrameCell[3][1] == 0.0
			 && md->its3DTilingIntoFrameCell[3][2] == 0.0,
			"received non-zero scroll position");
		
		theMazeSize = md->itsGameOf.Maze3D.itsSize;
		
		//	Scroll the slider to
		//
		//		x ==        0         in a left-to-right layout (with Help popover on the right)
		//	or
		//		x == theMazeSize - 1  in a right-to-left layout (with Help popover on the left)
		//
		//	(in integer maze coordinates).
		if ([[UIApplication sharedApplication] userInterfaceLayoutDirection] == UIUserInterfaceLayoutDirectionLeftToRight)
			theDesiredSliderX = 0;
		else
			theDesiredSliderX = (theMazeSize - 1);
		theOffset[0] = theDesiredSliderX - (signed int)md->itsGameOf.Maze3D.itsSlider.itsNode.p[0];

		//	Scroll the goal to y == theMazeSize - 1 (in integer maze coordinates).
		theOffset[1] = (theMazeSize - 1) - (signed int)md->itsGameOf.Maze3D.itsGoalNode.p[1];

		//	Scroll the slider to z == 0 (in integer maze coordinates).
		theOffset[2] = 0 - (signed int)md->itsGameOf.Maze3D.itsSlider.itsNode.p[2];
		
		md->its3DTilingIntoFrameCell[3][0] = (double)theOffset[0] / (double)theMazeSize;
		md->its3DTilingIntoFrameCell[3][1] = (double)theOffset[1] / (double)theMazeSize;
		md->its3DTilingIntoFrameCell[3][2] = (double)theOffset[2] / (double)theMazeSize;
		
		Normalize3DTilingIntoFrameCell(	md->its3DTilingIntoFrameCell,
										md->itsTopology);
	}

	[itsModel unlockModelData:&md];
}


#pragma mark -
#pragma mark TorusGamesGameChoiceDelegate

- (void)userDidChooseGame:(GameType)aNewGame
{
	[self changeGame:aNewGame];

	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)userDidCancelGameChoice
{
	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)changeGame:(GameType)aNewGame
{
	[self resetGameWithAnimationForPurpose:ResetWithNewGame value:aNewGame];
}

#pragma mark -
#pragma mark TorusGamesOptionsChoiceDelegate

- (void)userDidChooseOptionHumanVsComputer:(bool)aHumanVsComputer
{
	ModelData	*md	= NULL;

	//	ChangeHumanVsComputer() is always fast.  It never resets the game.
	[itsModel lockModelData:&md];
	ChangeHumanVsComputer(md, aHumanVsComputer);
	[itsModel unlockModelData:&md];

	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)userDidChooseOptionTopology:(TopologyType)aTopology
{
	[self resetGameWithAnimationForPurpose:ResetWithNewTopology value:aTopology];

	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)userDidChooseOptionDifficultyLevel:(unsigned int)aDifficultyLevel
{
	[self resetGameWithAnimationForPurpose:ResetWithNewDifficultyLevel value:aDifficultyLevel];

	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)userDidChooseOptionViewType:(ViewType)aViewType
{
	ModelData	*md	= NULL;

	[itsModel lockModelData:&md];
	ChangeViewType(md, aViewType);
	[itsModel unlockModelData:&md];

	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)userDidChooseOptionSoundEffects:(bool)aSoundEffects
{
	gPlaySounds = aSoundEffects;
	SetUserPrefBool(u"sound effects", gPlaySounds);

	[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)userDidCancelOptionsChoice
{
	[self dismissViewControllerAnimated:YES completion:nil];
}


#pragma mark -
#pragma mark TorusGamesLanguageChoiceDelegate

- (void)userDidChooseLanguage:(const Char16 [3])aTwoLetterLanguageCode
{
	NSString	*theTwoLetterLanguageCode;	//	user's requested language at the Torus Games level
	
	//	Switch to the requested language.
	[self changeLanguage:aTwoLetterLanguageCode];
	
	//	Convert aTwoLetterLanguageCode from a Char16 array
	//	(which the completion handler below can't retain)
	//	to an NSString (which the completion handler can retain).
	theTwoLetterLanguageCode = [NSString stringWithCharacters:aTwoLetterLanguageCode length:2];

	//	Dismiss the TorusGamesLanguageChoiceController and then
	//	-- after the presented view has been completely dismissed --
	//	warn the user about UniHan font issues, if relevant.
	[self dismissViewControllerAnimated:YES completion:^(void)
	{
		NSString			*thePreferredLanguage;	//	user's preferred language at the iOS level
		NSString			*theTitle		= nil,	//	initialization is unnecessary, but make our meaning clear
							*theMessage		= nil;
		UIAlertController	*theAlertController;

		//	Warn the user about UniHan font issues, if relevant.
		thePreferredLanguage = GetPreferredLanguage();
		if ( ! [theTwoLetterLanguageCode isEqualToString:thePreferredLanguage] )
		{
			if ([theTwoLetterLanguageCode isEqualToString:@"ja"])
			{
				theTitle	= @"For best kanji quality, please set the system language to Japanese";
				theMessage	= @"To ensure a Japanese system font, please go to the iOS Settings app > General > Language & Region > iPhone Language and select Japanese.  Otherwise kanji may appear in a Chinese system font.";
			}
			if ([theTwoLetterLanguageCode isEqualToString:@"zs"])
			{
				theTitle	= @"For best hanzi quality, please set the system language to Simplified Chinese";
				theMessage	= @"To ensure a Simplified Chinese system font, please go to the iOS Settings app > General > Language & Region > iPhone Language and select Simplified Chinese.  Otherwise hanzi may appear in a Traditional Chinese or Japanese system font.";
			}
			if ([theTwoLetterLanguageCode isEqualToString:@"zt"])
			{
				theTitle	= @"For best hanzi quality, please set the system language to Traditional Chinese";
				theMessage	= @"To ensure a Traditional Chinese system font, please go to the iOS Settings app > General > Language & Region > iPhone Language and select Traditional Chinese.  Otherwise hanzi may appear in a Simplified Chinese or Japanese system font.";
			}

#ifdef SCIENCE_CENTER_VERSION
#error Before releasing another Science Center version,			\
	test whether the "For best kanji quality" message			\
	works OK with it.  In particular, if that message appears	\
	and then the user walks away, what happens when				\
	the exhibits times out and resets itself?
#endif
			if (theTitle != nil && theMessage != nil)
			{
				theAlertController = [UIAlertController
					alertControllerWithTitle:	LocalizationNotNeeded(theTitle)
					message:					LocalizationNotNeeded(theMessage)
					preferredStyle:				UIAlertControllerStyleAlert];

				[theAlertController addAction:[UIAlertAction
					actionWithTitle:	LocalizationNotNeeded(@"OK")
					style:				UIAlertActionStyleDefault
					handler:			nil]];

				[self	presentViewController:	theAlertController
						animated:				YES
						completion:				nil];
			}
		}
	}];
}

- (void)userDidCancelLanguageChoice
{
	//	-userDidCancelLanguageChoice gets called
	//	only when the user taps the Cancel button,
	//	not when the user taps outside a popover.

	[self dismissViewControllerAnimated:YES completion:nil];
}


#pragma mark -
#pragma mark language

- (bool)changeLanguage:(const Char16 [3])aTwoLetterLanguageCode
{
	ModelData	*md	= NULL;
	GameType	theGame;

	if ( ! IsCurrentLanguage(aTwoLetterLanguageCode) )
	{
		//	Record the new language code and initialize a dictionary.
		SetCurrentLanguage(aTwoLetterLanguageCode);

		//	Update the toolbar for the new language.
		[self refreshToolbarContent];
		
		//	Reset the game only if necessary.

		[itsModel lockModelData:&md];
		theGame = md->itsGame;
		[itsModel unlockModelData:&md];

		switch (theGame)
		{
			case Game2DCrossword:

				[itsModel lockModelData:&md];
				ResetGame(md);
				if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
					[((TorusGamesGraphicsViewiOS *)itsMainView) refreshRendererTexturesWithModelData:md];
				[itsModel unlockModelData:&md];

				[itsKeyboardView refreshWithLanguage:GetCurrentLanguage()];

				break;

			case Game2DWordSearch:

				[itsModel lockModelData:&md];
				ResetGame(md);
				if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
					[((TorusGamesGraphicsViewiOS *)itsMainView) refreshRendererTexturesWithModelData:md];
				[itsModel unlockModelData:&md];

				break;
			
			case Game2DPool:
			case Game2DApples:
			case Game3DMaze:

				//	No need to disturb the Pool game, Apples game or 3D Maze in progress.
				//	Just refresh the current status or instructions message in the new language.
				[itsModel lockModelData:&md];
				RefreshStatusMessageText(md);
				[itsModel unlockModelData:&md];

				break;
			
			default:
				//	Do nothing.
				break;
		}

		//	Refresh itsMessageView's centering.
		[self refreshMessageCentering];
		
		//	Do *not* try to change the user interface layout direction.
		//	iOS 9 gives us a left-to-right or right-to-left layout
		//	according to the underlying iOS system language,
		//	with no effort required on our part, so let's stick with that.
		//	(Even in the pre-iOS-9 days, when I tried to manually reflect
		//	KaleidoPaint's user interface, it was tedious to code
		//	and the results were far from complete.)
		
		return true;
	}
	else
	{
		return false;
	}
}


#pragma mark -
#pragma mark message centering

- (void)refreshMessageCentering
{
	ModelData	*md	= NULL;
	GameType	theGame;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	//	The Japanese Word Search list looks best left aligned,
	//	so that the characters align vertically.
	//	For all other games and languages, center aligned text looks best.
	if (theGame == Game2DWordSearch && IsCurrentLanguage(u"ja"))
		[itsMessageView setTextAlignment:NSTextAlignmentLeft];
	else
		[itsMessageView setTextAlignment:NSTextAlignmentCenter];
}


#pragma mark -
#pragma mark TorusGamesHelpChoiceDelegate

- (void)dismissHelp
{
	[self dismissViewControllerAnimated:YES completion:nil];
}

- (bool)practiceBoardIsSelected
{
	ModelData	*md	= NULL;
	GameType	theGame;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	return theGame == Game2DIntro;
}

- (void)selectPracticeBoard
{
	[self changeGame:Game2DIntro];

	//	In a horizontally compact layout, where we expect
	//	the Help panel to completely cover the main view,
	//	dismiss the Help panel so the user may see
	//	the practice board that s/he just selected.
	//
	//	In a horizontally regular layout, where we expect
	//	the Help panel to be presented as a popover,
	//	leave the Help panel up until the user explicitly closes it.
	//
	if ([[self traitCollection] horizontalSizeClass] == UIUserInterfaceSizeClassCompact)
		[self dismissViewControllerAnimated:YES completion:nil];
}

- (void)requestGame:(GameType)aNewGame
{
	ModelData	*md	= NULL;
	bool		theChangeGameFlag;

	//	For the most part, we want to honor the Help chooser's
	//	request for a new game.  The only exception is that
	//	the Wraparound Universe text, midway through, asks
	//	the user to switch to the Jigsaw puzzle, so we don't
	//	want to force the app back to the Intro when the user
	//	re-opens the Wraparound Universe page.
	[itsModel lockModelData:&md];
	theChangeGameFlag = (aNewGame != Game2DIntro || md->itsGame != Game2DJigsaw);
	[itsModel unlockModelData:&md];

	if (theChangeGameFlag)
		[self resetGameWithAnimationForPurpose:ResetWithNewGame value:aNewGame];
}


#pragma mark -
#pragma mark TorusGamesKeyboardViewDelegate

- (void)userTappedKey:(unichar)aCharacter
{
	ModelData			*md					= NULL;
	TitledErrorMessage	*theErrorMessage	= NULL;
	bool				theGameIsOver;
	NSString			*theTitle,
						*theMessage;
	UIAlertController	*theAlertController;
	UIViewController	*thePresentingController;

	[itsModel lockModelData:&md];

	//	CrosswordCharacterInput() ignores characters typed after the game is over.
	theErrorMessage = CharacterInput(md, (Char16)aCharacter);
	
	//	Request texture updates before unlocking the ModelData, to ensure that
	//	the animation thread finds the ModelData and the textures
	//	in a consistent state.
	if ([itsMainView isKindOfClass:[TorusGamesGraphicsViewiOS class]])	//	should never fail
		[((TorusGamesGraphicsViewiOS *)itsMainView)
			refreshRendererTexturesForCharacterInputWithModelData:md];

	//	Don't let SoundCrosswordKeypress compete with SoundCrosswordComplete.
	theGameIsOver = md->itsGameIsOver;

	[itsModel unlockModelData:&md];

	if ( ! theGameIsOver )
		EnqueueSoundRequest(u"CrosswordKeypress.mid");
	
	if (theErrorMessage != NULL)
	{
		theTitle	= GetNSStringFromZeroTerminatedString(theErrorMessage->itsTitle);
		theMessage	= GetNSStringFromZeroTerminatedString(theErrorMessage->itsMessage);

		theAlertController = [UIAlertController
			alertControllerWithTitle:	theTitle
			message:					theMessage
			preferredStyle:				UIAlertControllerStyleAlert];

		[theAlertController addAction:[UIAlertAction
			actionWithTitle:	LocalizationNotNeeded(@"OK")
			style:				UIAlertActionStyleDefault
			handler:			nil]];

		//	Walk up the stack of already presented view controllers (if any)
		//	to reach the topmost view controller.
		//
		//		Warning:  This approach is less than perfect,
		//		because if the code that posted the error message
		//		happens to dismiss thePresentingController,
		//		then theAlertController will get silently dismissed
		//		along with it, and the user will see the error message
		//		for only a small fraction of a second.
		//
		thePresentingController = self;
		while ([thePresentingController presentedViewController] != nil)
		{
			thePresentingController = [thePresentingController presentedViewController];
		}

		[thePresentingController
			presentViewController:	theAlertController
			animated:				YES
			completion:				nil];
	}

#ifdef SCIENCE_CENTER_VERSION
	[self userIsStillPlaying];
#endif
}


#pragma mark -
#pragma mark TorusGamesMatteViewDelegate

- (void)matteViewDidSetLayoutOrientation:(MatteViewLayoutOrientation)aMatteViewLayoutOrientation
{
	ModelData	*md	= NULL;

	[itsModel lockModelData:&md];
	
	md->itsChinesePoemsWantColumnFormatting
		= (aMatteViewLayoutOrientation == MatteViewLandscapeLayout);

	[itsModel unlockModelData:&md];
}


#ifdef SCIENCE_CENTER_VERSION

#pragma mark -
#pragma mark Inactivity timeout


- (void)userIsStillPlaying
{
	itsUserIsActivelyPlaying = true;
}

- (void)inactivityTimerFired:(id)sender
{
	ModelData			*md	= NULL;
	GameType			theGame;
	bool				theHumanVsComputer;
	TopologyType		theTopology;
	unsigned int		theDifficultyLevel;
	UIAlertController	*theTimeoutMessageController;
	
	GEOMETRY_GAMES_ASSERT(sender == itsInactivityTimer, "Unknown timer");
	
	if (itsUserIsActivelyPlaying)
	{
		//	As long as the visitor is still playing,
		//	ignore the timer and let the visitor continue.
		
		//	Reset itsUserIsActivelyPlaying to false,
		//	so the next time the timer fires we'll know
		//	whether or not the user is still playing.
		itsUserIsActivelyPlaying = false;
	}
	else
	if ( ! itsKeepPlayingMessageIsUp )
	{
		//	The inactivity timer typically fires when a group of visitors
		//	finishes playing with the app and goes on to the other exhibits,
		//	but we also want to allow for the possibility that some visitors
		//	are discussing, say, a torus chess move at great length,
		//	and want to keep playing.  So put up an action sheet asking
		//	whether the visitors want to keep playing.  If they ignore it
		//	until the time fires again, then reset the exhibit.

		[itsModel lockModelData:&md];
		theGame				= md->itsGame;
		theHumanVsComputer	= md->itsHumanVsComputer;
		theTopology			= md->itsTopology;
		theDifficultyLevel	= md->itsDifficultyLevel;
		[itsModel unlockModelData:&md];

		//	If the exhibit is in its pristine state...
		if
		(
			IsCurrentLanguage(u"en")
		 && theGame				== GameNone
		 && theHumanVsComputer	== true
		 && theTopology			== Topology2DTorus
		 && theDifficultyLevel	== 1
		 && [[[self presentedViewController] popoverPresentationController] barButtonItem] == itsChangeGameButton
		)
		{
			//	...then wait silently until the next time itsInactivityTimer fires.
		}
		else
		{
			//	If the exhibit isn't in its pristine state,
			//	then a group of visitors either
			//
			//		- has finished playing with the app
			//			and moved onward to the other exhibits
			//	or
			//		- is, for example, discussing a torus chess move
			//			at great length, and wants to keep playing.
			//
			//	To gracefully handle the second possibility, put up
			//	a dialog asking whether the visitors want to keep playing.
			//	If the visitors ignore it until the next time the timer fires,
			//	then reset the exhibit.

			//	Technical Note:  If some other presented view is already up,
			//	the UIKit will send a message to the console saying
			//
			//		Warning: Attempt to present <UIAlertController: 0x........>
			//		on <TorusGamesRootController: 0x........> which
			//		is already presenting <UINavigationController: ........>
			//
			//	and then not present theTimeoutMessageController.
			//	So let's dismiss any pre-existing presented view controller
			//	before putting up the new one.
			//
			if ([self presentedViewController] != nil)
				[self dismissViewControllerAnimated:YES completion:nil];

			theTimeoutMessageController = [UIAlertController
											alertControllerWithTitle:	nil
											message:					nil
											preferredStyle:				UIAlertControllerStyleAlert];
			
			[theTimeoutMessageController addAction:[UIAlertAction
				actionWithTitle:	LocalizationNotNeeded(@"Reset Torus Games")	//	not localized
				style:				UIAlertActionStyleDestructive
				handler:^(UIAlertAction *action)
				{
					UNUSED_PARAMETER(action);

					//	Strong reference to self is harmless here
					//	because TorusGamesRootController lives
					//	as long as the app does.

					//	Dismiss the "Keep playing?" message.
					[self dismissViewControllerAnimated:YES completion:nil];
					itsKeepPlayingMessageIsUp	= false;
					itsUserIsActivelyPlaying	= true;
					
					//	Reset the exhibit.
					[self resetExhibit];
				}]];

			[theTimeoutMessageController addAction:[UIAlertAction
				actionWithTitle:	LocalizationNotNeeded(@"Keep playing")	//	not localized
				style:				UIAlertActionStyleDefault
				handler:^(UIAlertAction *action)
				{
					UNUSED_PARAMETER(action);

					//	Strong reference to self is harmless here
					//	because TorusGamesRootController lives
					//	as long as the app does.

					//	Dismiss the "Keep playing?" message,
					//	but leave the game as it is.
					[self dismissViewControllerAnimated:YES completion:nil];
					itsKeepPlayingMessageIsUp	= false;
					itsUserIsActivelyPlaying	= true;
				}]];
			
			[self presentViewController:theTimeoutMessageController animated:YES completion:nil];
			itsKeepPlayingMessageIsUp = true;
		}
	}
	else	//	itsKeepPlayingMessageIsUp
	{
		//	The visitors have left, so...
		
		//	...dismiss the "Keep playing?" message
		[self dismissViewControllerAnimated:YES completion:nil];
		itsKeepPlayingMessageIsUp	= false;
		itsUserIsActivelyPlaying	= true;
		
		//	...and reset the exhibit.
		[self resetExhibit];
	}
}

- (void)resetExhibit
{
	ModelData	*md	= NULL;

	//	Dismiss the system keyboard if it's up
	//	(the system keyboard is used for Chinese crossword puzzles).
	[itsKeyboardView endEditing:YES];
	
	//	Restore default settings.
	[self changeGame:GameNone];	//	will release game-specific Metal textures
	[self changeLanguage:u"en"];
	[itsModel lockModelData:&md];
	ChangeHumanVsComputer(md, true);
	ChangeTopology(md, Topology2DTorus);
	ChangeDifficultyLevel(md, 2);
	gd->itsPreparedTextures = false;
	[itsModel unlockModelData:&md];

	//	Show the Choose Game panel, and then
	//	wait patiently for a new visitor to arrive.
	[self simulateChangeGameButtonTap];
}


#endif


@end
